/*
    Sharpen Feathering - A GIMP plugin
    Copyright (C) 2003  Paul Francis Harrison

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <unistd.h>
#include <stdlib.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

typedef int bool;
const bool true = 1, false = 0;

/* Talking to GIMP stuff */

static void query();
static void run(const gchar *name,
                gint nparams,
		const GimpParam *param,
		gint *nreturn_vals,
		GimpParam **return_vals);

GimpPlugInInfo PLUG_IN_INFO = {
  NULL, /* init_proc */
  NULL, /* quit_proc */
  query, /* query_proc */
  run, /* run_proc */
};

/* Macro to define the usual plugin main function */
MAIN()

/* Add capabilities of this plugin to Procedural DataBase */
static void query(void) {
  static GimpParamDef args[] = {
    { GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_IMAGE, "image", "Input image" },
    { GIMP_PDB_DRAWABLE, "drawable", "Input drawable" }
  };

  GimpParamDef *return_vals = NULL;
  gint nargs = sizeof(args)/sizeof(args[0]);
  gint nreturn_vals = 0;

  gimp_install_procedure("plug-in-sharpfeather",
			 "Maintain sharpness of feathering around an object by increasing "
			 "the contrast of areas with alpha less than 1.",
			 "",
			 "Paul Francis Harrison",
			 "Paul Francis Harrison",
			 "2003",
			 "<Image>/Filters/Enhance/Sharpen Feathering",
			 "RGBA, GRAYA",
			 GIMP_PLUGIN,
			 nargs, nreturn_vals,
			 args, return_vals);
}


static void run(const gchar *name,
                gint nparams,
		const GimpParam *param,
		gint *nreturn_vals,
		GimpParam **return_vals) {
  GimpDrawable *drawable;
  gint32 drawable_id;
  GimpPixelRgn srcRegion, destRegion;
  gpointer pr;
  int i,j,x1,y1,x2,y2,width,height;
  double avg_color[3] = {0.0,0.0,0.0}, avg_color_div = 0.0;

  static GimpParam values[1];
  
  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_SUCCESS;

  /* Get drawable */
  drawable_id = param[2].data.d_drawable;
  drawable = gimp_drawable_get(drawable_id);

  if ((!gimp_drawable_is_rgb(drawable_id) &&
       !gimp_drawable_is_gray(drawable_id)) ||
       !gimp_drawable_has_alpha(drawable_id)) {
    gimp_drawable_detach(drawable);
    values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
    return;
  }

  /* Fetch the image to be warped, munged, crunched, twisted... */
  gimp_drawable_mask_bounds(drawable_id,&x1,&y1,&x2,&y2);
  width = x2-x1;
  height = y2-y1;
  
  /* Do it */
  
  /* Calculate mean color */
  gimp_pixel_rgn_init(&srcRegion, drawable,x1,y1,width,height,FALSE,FALSE);

  for(pr = gimp_pixel_rgns_register(1,&srcRegion);
      pr != NULL;
      pr = gimp_pixel_rgns_process(pr)) 
    for(i=0;i<srcRegion.w*srcRegion.h*srcRegion.bpp;i+=srcRegion.bpp) {
      double weight = srcRegion.data[i+srcRegion.bpp-1]/255.0;
      avg_color_div += weight;
      for(j=0;j<srcRegion.bpp-1;j++)
        avg_color[j] += srcRegion.data[i+j] * weight;
    }

  if (avg_color_div != 0.0)
    for(i=0;i<srcRegion.bpp-1;i++)
      avg_color[i] /= avg_color_div;
      
  /* Perform contrast stretch */
  gimp_pixel_rgn_init(&srcRegion, drawable,x1,y1,width,height,FALSE,FALSE);
  gimp_pixel_rgn_init(&destRegion,drawable,x1,y1,width,height,TRUE,TRUE);

  for(pr = gimp_pixel_rgns_register(2,&srcRegion,&destRegion);
      pr != NULL;
      pr = gimp_pixel_rgns_process(pr))
    for(i=0;i<srcRegion.w*srcRegion.h*srcRegion.bpp;i+=srcRegion.bpp) 
      if (srcRegion.data[i+srcRegion.bpp-1] == 0 ||
          srcRegion.data[i+srcRegion.bpp-1] == 255) {
        for(j=0;j<srcRegion.bpp;j++)
	  destRegion.data[i+j] = srcRegion.data[i+j];
      } else {
	double factor = sqrt( 510.0 / srcRegion.data[i+srcRegion.bpp-1] - 1.0 );
        for(j=0;j<srcRegion.bpp-1;j++) {
	  int value = (int)( (srcRegion.data[i+j]-avg_color[j])*factor + avg_color[j] + 0.5 );
	  if (value < 0) value = 0;
	  if (value > 255) value = 255;
	  destRegion.data[i+j] = (guchar)value;
	}
        
	/* Alternative implementation
	double alpha = srcRegion.data[i+srcRegion.bpp-1] / 255.0;
	double old_sd = sqrt(2*alpha*alpha-2*alpha+1);
	double new_alpha = 1 - (1-alpha)/old_sd;
	double contrast_correction = alpha / (old_sd * new_alpha);
	double avg_correction = (1.0 - 1.0/old_sd) / new_alpha;

	destRegion.data[i+srcRegion.bpp-1] = (guchar)(new_alpha*255.0);
        for(j=0;j<srcRegion.bpp-1;j++) {
	  int value = (int)( srcRegion.data[i+j]*contrast_correction + avg_color[j]*avg_correction + 0.5 );
	  if (value < 0) value = 0;
	  if (value > 255) value = 255;
	  destRegion.data[i+j] = (guchar)value;
	}*/
      }
    
  /* Voodoo to update actual image */
  gimp_drawable_flush(drawable);
  gimp_drawable_merge_shadow(drawable_id,TRUE);
  gimp_drawable_update(drawable_id,x1,y1,width,height);

  /* Unget drawable */
  gimp_drawable_detach(drawable);
  
  if (param[0].data.d_int32 != GIMP_RUN_NONINTERACTIVE)
    gimp_displays_flush();
}

