
; Patch-splat texture synthesis
; (Fast tilable texture synthesis script)

;    Copyright (C) 2003  Paul Francis Harrison
 
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
 
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
 
;    You should have received a copy of the GNU General Public License
;    along with this program; if not, write to the Free Software
;    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

; Copy a random rectangle to clipboard
(define (grab-patch img layer patch-size patch-border)
  (let*
    (
      (width    (car (gimp-drawable-width layer)))
      (height   (car (gimp-drawable-height layer)))
    )

    (gimp-rect-select img
      (+ patch-border (rand (- width  (* patch-border 2) patch-size)))
      (+ patch-border (rand (- height (* patch-border 2) patch-size)))
      patch-size patch-size 2 1 (* patch-border 2))
    (gimp-edit-copy layer)
  )
)

; Paste the clipboard and apply Sharpen Feathering
(define (put-patch img layer x y use-sharp)
  (let*
    (
      (floater (car (gimp-edit-paste layer FALSE)))
    )
    (gimp-layer-set-offsets floater x y)
    (if (= use-sharp TRUE) (plug-in-sharpfeather 1 img floater))
    (gimp-floating-sel-anchor floater)
  )
)

; Shuffle a list randomly (hacky)
(define (shuffle old)
  (let*
    (
      (new '())
    )

    (while (not (= old '()))
      (if (= (rand 2) 1)
        (set! new (cons (car old) new))
	(set! new (append new (list (car old))))
      )
      (set! old (cdr old))
    )

    new
  )
)

(define (script-fu-patch-synthesize img layer out-width out-height patch-proportion patch-feather-proportion use-sharp)
  (let*
    (
      (width     (car (gimp-drawable-width layer)))
      (height    (car (gimp-drawable-height layer)))
      (type      (car (gimp-drawable-type layer)))
      (base-type (car (gimp-image-base-type img)))
      (new-img   (car (gimp-image-new out-width out-height base-type)))
      (new-layer (car (gimp-layer-new new-img out-width out-height type "Background" 100 0)))

      (patch-size (* (min width height out-width out-height) patch-proportion 0.5))
      (patch-border (* patch-size patch-feather-proportion 0.25))
      (patch-jitter (/ patch-size 8))
      (patch-total  (+ patch-size patch-border patch-border))

      (xx 0)
      (yy 0)
      (old-selection 0)
      (point-list '())
    )

    (gimp-image-add-layer new-img new-layer 0)
    (gimp-edit-clear new-layer)
    (gimp-display-new new-img)
    (gimp-displays-flush)

    (gimp-image-undo-freeze img)
    (gimp-image-undo-disable new-img)

    (set! old-selection (car (gimp-selection-save img)))

    (while (< xx out-width)
      (set! yy 0)
      (while (< yy out-height)
        (set! point-list (cons (cons xx yy) point-list))
        (set! yy (+ yy (- patch-size patch-border patch-border patch-jitter)))
      )
      (set! xx (+ xx (- patch-size patch-border patch-border patch-jitter)))
    )

    (set! point-list (shuffle point-list))
    (set! point-list (shuffle point-list))
    (set! point-list (shuffle point-list))
    (set! point-list (shuffle point-list))

    (while (not (= point-list nil))
      (grab-patch img layer patch-size patch-border)

      (let*
	(
	  (x (+ (caar point-list) (rand patch-jitter)))
	  (y (+ (cdar point-list) (rand patch-jitter)))
	)

	(put-patch new-img new-layer x y use-sharp)
	(if (>= x (- out-width patch-total))
	  (put-patch new-img new-layer (- x out-width) y use-sharp)
	)
	(if (>= y (- out-height patch-total))
	  (put-patch new-img new-layer x (- y out-height) use-sharp)
	)
	(if 
	  (and
	    (>= x (- out-width patch-total))
	    (>= y (- out-height patch-total))
	  )    
	  (put-patch new-img new-layer (- x out-width) (- y out-height) use-sharp)
	)
      )

      (if (= (fmod (length point-list) 8) 7) (gimp-displays-flush))

      (set! point-list (cdr point-list))
    )

    (gimp-selection-load old-selection)
    (gimp-image-remove-channel img old-selection)
    
    (gimp-image-undo-enable new-img)
    (gimp-image-undo-thaw img)

    (gimp-displays-flush)
  )
)

(script-fu-register "script-fu-patch-synthesize"
                    "<Image>/Script-Fu/Render/Synthesize Texture..."
		    "Create a tilable texture of arbitrary size from a sample, by simply copying random patches of the sample. The edges of each patch are disguised using the 'Sharpen Feathering' plug-in."
		    "Paul Harrison (pfh@csse.monash.edu.au)"
		    "Paul Harrison"
		    "18/3/2003"
		    "RGB GRAY"
		    SF-IMAGE "Input Image" 0
		    SF-DRAWABLE "Input Layer" 0
		    SF-VALUE "Output width" "300"
		    SF-VALUE "Output height" "300"
		    SF-ADJUSTMENT "Patch size" '(0.6 0.1 1.0 0.01 0.1 1 0)
		    SF-ADJUSTMENT "Patch feathering" '(0.5 0.0 1.0 0.01 0.1 1 0)
		    SF-TOGGLE "Use feather sharpen" 1
)			
